<?php
session_start();
if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit;
}
require_once '../config/db.php';
$conn = getDB();

// Get filter and search parameters
$status_filter = isset($_GET['status']) ? $_GET['status'] : '';
$section_filter = isset($_GET['section']) ? $_GET['section'] : '';
$search = isset($_GET['search']) ? trim($_GET['search']) : '';

// Pagination parameters
$page = isset($_GET['page']) ? max(1, intval($_GET['page'])) : 1;
$records_per_page = 10;
$offset = ($page - 1) * $records_per_page;

// Base query for counting total records
$count_sql = "SELECT COUNT(DISTINCT a.application_id) as total 
              FROM applications a 
              LEFT JOIN payments p ON a.application_id = p.application_id 
              LEFT JOIN admission_status s ON a.application_id = s.application_id";

// Base query for fetching records
$sql = "SELECT a.*, p.status AS payment_status, s.status AS admission_status, s.score 
        FROM applications a 
        LEFT JOIN payments p ON a.application_id = p.application_id 
        LEFT JOIN admission_status s ON a.application_id = s.application_id";

// Build WHERE clauses
$where_clauses = [];
$params = [];
$types = '';

if ($status_filter) {
    $where_clauses[] = "s.status = ?";
    $params[] = $status_filter;
    $types .= 's';
}

if ($section_filter) {
    $where_clauses[] = "a.section = ?";
    $params[] = $section_filter;
    $types .= 's';
}

if ($search) {
    $where_clauses[] = "(a.full_name LIKE ? OR a.application_id LIKE ? OR a.email LIKE ? OR a.phone LIKE ?)";
    $search_param = '%' . $search . '%';
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $params[] = $search_param;
    $types .= 'ssss';
}

// Add WHERE clause if needed
if (!empty($where_clauses)) {
    $where_sql = " WHERE " . implode(' AND ', $where_clauses);
    $count_sql .= $where_sql;
    $sql .= $where_sql;
}

// Get total count
$count_stmt = $conn->prepare($count_sql);
if (!empty($params)) {
    $count_stmt->bind_param($types, ...$params);
}
$count_stmt->execute();
$total_records = $count_stmt->get_result()->fetch_assoc()['total'];
$total_pages = ceil($total_records / $records_per_page);

// Add ORDER BY and LIMIT
$sql .= " ORDER BY a.submitted_at DESC LIMIT ? OFFSET ?";
$params[] = $records_per_page;
$params[] = $offset;
$types .= 'ii';

// Execute main query
$stmt = $conn->prepare($sql);
if (!empty($params)) {
    $stmt->bind_param($types, ...$params);
}
$stmt->execute();
$result = $stmt->get_result();

// Build query string for pagination links
function buildQueryString($exclude = []) {
    $params = $_GET;
    foreach ($exclude as $key) {
        unset($params[$key]);
    }
    return http_build_query($params);
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
    <style>
        .search-container {
            background: white;
            border-radius: 8px;
            padding: 20px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
            margin-bottom: 20px;
        }
        .pagination-info {
            color: #6c757d;
            font-size: 0.9em;
        }
        .table-container {
            background: white;
            border-radius: 8px;
            overflow: hidden;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        .page-link {
            color: #495057;
        }
        .page-link:hover {
            color: #495057;
        }
        .page-item.active .page-link {
            background-color: #495057;
            border-color: #495057;
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <?php include'sidebar.php'; ?>

        <!-- Main Content -->
        <div class="main-content">
            <div class="container-fluid py-4">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Manage Applications</h2>
                    <div class="btn-group">
                        <button type="button" class="btn btn-success dropdown-toggle" data-bs-toggle="dropdown">
                            <i class="bi bi-download"></i> Export Admitted
                        </button>
                        <ul class="dropdown-menu">
                            <li>
                                <a class="dropdown-item" href="../api/export_admitted.php?section=primary">
                                    <i class="bi bi-people"></i> Primary Section
                                </a>
                            </li>
                            <li>
                                <a class="dropdown-item" href="../api/export_admitted.php?section=junior">
                                    <i class="bi bi-people"></i> Junior Section
                                </a>
                            </li>
                        </ul>
                    </div>
                </div>

                <!-- Search and Filter Section -->
                <div class="search-container">
                    <form method="GET" class="row g-3 align-items-end">
                        <div class="col-md-4">
                            <label for="search" class="form-label">Search Applications</label>
                            <div class="input-group">
                                <span class="input-group-text"><i class="bi bi-search"></i></span>
                                <input type="text" class="form-control" id="search" name="search" 
                                       placeholder="Name, ID, Email, Phone..." value="<?php echo htmlspecialchars($search); ?>">
                            </div>
                        </div>
                        <div class="col-md-3">
                            <label for="status" class="form-label">Status Filter</label>
                            <select class="form-select" id="status" name="status">
                                <option value="">All Statuses</option>
                                <option value="pending" <?php echo $status_filter === 'pending' ? 'selected' : ''; ?>>Pending</option>
                                <option value="under_review" <?php echo $status_filter === 'under_review' ? 'selected' : ''; ?>>Under Review</option>
                                <option value="accepted" <?php echo $status_filter === 'accepted' ? 'selected' : ''; ?>>Accepted</option>
                                <option value="rejected" <?php echo $status_filter === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label for="section" class="form-label">Section Filter</label>
                            <select class="form-select" id="section" name="section">
                                <option value="">All Sections</option>
                                <option value="primary" <?php echo $section_filter === 'primary' ? 'selected' : ''; ?>>Primary</option>
                                <option value="junior" <?php echo $section_filter === 'junior' ? 'selected' : ''; ?>>Junior</option>
                            </select>
                        </div>
                        <div class="col-md-2">
                            <button type="submit" class="btn btn-primary w-100">
                                <i class="bi bi-funnel"></i> Filter
                            </button>
                        </div>
                    </form>
                    
                    <?php if ($search || $status_filter || $section_filter): ?>
                    <div class="mt-3">
                        <a href="applicants.php" class="btn btn-outline-secondary btn-sm">
                            <i class="bi bi-x-circle"></i> Clear Filters
                        </a>
                        <span class="ms-2 text-muted">
                            Active filters: 
                            <?php 
                            $active_filters = [];
                            if ($search) $active_filters[] = "Search: '$search'";
                            if ($status_filter) $active_filters[] = "Status: " . ucfirst($status_filter);
                            if ($section_filter) $active_filters[] = "Section: " . ucfirst($section_filter);
                            echo implode(', ', $active_filters);
                            ?>
                        </span>
                    </div>
                    <?php endif; ?>
                </div>

                <!-- Results Info and Pagination Info -->
                <div class="d-flex justify-content-between align-items-center mb-3">
                    <div class="pagination-info">
                        Showing <?php echo $offset + 1; ?> to <?php echo min($offset + $records_per_page, $total_records); ?> 
                        of <?php echo $total_records; ?> applications
                    </div>
                    <div class="pagination-info">
                        Page <?php echo $page; ?> of <?php echo $total_pages; ?>
                    </div>
                </div>

                <!-- Applications Table -->
                <div class="table-container">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Section</th>
                                    <th>Age</th>
                                    <th>Payment</th>
                                    <th>Status</th>
                                    <th>Score</th>
                                    <th>Submitted</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if ($result->num_rows > 0): ?>
                                    <?php while ($row = $result->fetch_assoc()): ?>
                                        <tr>
                                            <td><span class="badge bg-light text-dark"><?php echo $row['application_id']; ?></span></td>
                                            <td>
                                                <div>
                                                    <strong><?php echo htmlspecialchars($row['full_name']); ?></strong>
                                                    <br><small class="text-muted"><?php echo htmlspecialchars($row['email']); ?></small>
                                                </div>
                                            </td>
                                            <td><?php echo ucfirst($row['section']); ?></td>
                                            <td><?php echo $row['age']; ?></td>
                                            <td>
                                                <span class="badge bg-<?php echo $row['payment_status'] === 'successful' ? 'success' : 'warning'; ?>">
                                                    <?php echo ucfirst($row['payment_status'] ?: 'Pending'); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <span class="badge bg-<?php 
                                                    echo match($row['admission_status']) {
                                                        'accepted' => 'success',
                                                        'rejected' => 'danger',
                                                        'under_review' => 'info',
                                                        default => 'secondary'
                                                    };
                                                ?>">
                                                    <?php echo ucfirst($row['admission_status'] ?: 'Pending'); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <?php if ($row['score']): ?>
                                                    <div class="progress" style="height: 20px; min-width: 60px;">
                                                        <div class="progress-bar" role="progressbar" style="width: <?php echo $row['score']; ?>%" 
                                                             aria-valuenow="<?php echo $row['score']; ?>" aria-valuemin="0" aria-valuemax="100">
                                                            <?php echo $row['score']; ?>%
                                                        </div>
                                                    </div>
                                                <?php else: ?>
                                                    <span class="text-muted">N/A</span>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <small><?php echo date('M j, Y', strtotime($row['submitted_at'])); ?></small>
                                            </td>
                                            <td>
                                                <div class="dropdown">
                                                    <button class="btn btn-sm btn-outline-primary dropdown-toggle" type="button" 
                                                            data-bs-toggle="dropdown" aria-expanded="false">
                                                        Actions
                                                    </button>
                                                    <ul class="dropdown-menu">
                                                        <li>
                                                            <a class="dropdown-item" href="#" data-bs-toggle="modal" 
                                                               data-bs-target="#viewModal" 
                                                               onclick="viewApplication('<?php echo $row['application_id']; ?>')">
                                                                <i class="bi bi-eye"></i> View Details
                                                            </a>
                                                        </li>
                                                        <li>
                                                            <a class="dropdown-item" href="#" onclick="updateStatus('<?php echo $row['application_id']; ?>', 'accepted')">
                                                                <i class="bi bi-check-circle"></i> Accept
                                                            </a>
                                                        </li>
                                                        <li>
                                                            <a class="dropdown-item" href="#" onclick="updateStatus('<?php echo $row['application_id']; ?>', 'rejected')">
                                                                <i class="bi bi-x-circle"></i> Reject
                                                            </a>
                                                        </li>
                                                        <li>
                                                            <a class="dropdown-item" href="#" onclick="promptScore('<?php echo $row['application_id']; ?>')">
                                                                <i class="bi bi-percent"></i> Set Score
                                                            </a>
                                                        </li>
                                                        <li>
                                                            <a class="dropdown-item" href="#" onclick="uploadLetter('<?php echo $row['application_id']; ?>')">
                                                                <i class="bi bi-file-earmark-pdf"></i> Upload Letter
                                                            </a>
                                                        </li>
                                                        <li>
                                                            <a class="dropdown-item" href="../uploads/<?php echo $row['photograph']; ?>" download>
                                                                <i class="bi bi-download"></i> Download Photo
                                                            </a>
                                                        </li>
                                                    </ul>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endwhile; ?>
                                <?php else: ?>
                                    <tr>
                                        <td colspan="9" class="text-center py-4">
                                            <div class="text-muted">
                                                <i class="bi bi-inbox display-1"></i>
                                                <p class="mt-2">No applications found matching your criteria.</p>
                                                <?php if ($search || $status_filter || $section_filter): ?>
                                                    <a href="applicants.php" class="btn btn-outline-primary">Clear Filters</a>
                                                <?php endif; ?>
                                            </div>
                                        </td>
                                    </tr>
                                <?php endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Pagination -->
                <?php if ($total_pages > 1): ?>
                <nav aria-label="Applications pagination" class="mt-4">
                    <ul class="pagination justify-content-center">
                        <!-- Previous Button -->
                        <li class="page-item <?php echo $page <= 1 ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?<?php echo buildQueryString(['page']); ?>&page=<?php echo $page - 1; ?>" 
                               <?php echo $page <= 1 ? 'tabindex="-1" aria-disabled="true"' : ''; ?>>
                                <i class="bi bi-chevron-left"></i> Previous
                            </a>
                        </li>

                        <?php
                        // Calculate page range to show
                        $start_page = max(1, $page - 2);
                        $end_page = min($total_pages, $page + 2);
                        
                        // Show first page if not in range
                        if ($start_page > 1): ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?php echo buildQueryString(['page']); ?>&page=1">1</a>
                            </li>
                            <?php if ($start_page > 2): ?>
                                <li class="page-item disabled"><span class="page-link">...</span></li>
                            <?php endif; ?>
                        <?php endif; ?>

                        <!-- Page numbers -->
                        <?php for ($i = $start_page; $i <= $end_page; $i++): ?>
                            <li class="page-item <?php echo $i == $page ? 'active' : ''; ?>">
                                <a class="page-link" href="?<?php echo buildQueryString(['page']); ?>&page=<?php echo $i; ?>"><?php echo $i; ?></a>
                            </li>
                        <?php endfor; ?>

                        <!-- Show last page if not in range -->
                        <?php if ($end_page < $total_pages): ?>
                            <?php if ($end_page < $total_pages - 1): ?>
                                <li class="page-item disabled"><span class="page-link">...</span></li>
                            <?php endif; ?>
                            <li class="page-item">
                                <a class="page-link" href="?<?php echo buildQueryString(['page']); ?>&page=<?php echo $total_pages; ?>"><?php echo $total_pages; ?></a>
                            </li>
                        <?php endif; ?>

                        <!-- Next Button -->
                        <li class="page-item <?php echo $page >= $total_pages ? 'disabled' : ''; ?>">
                            <a class="page-link" href="?<?php echo buildQueryString(['page']); ?>&page=<?php echo $page + 1; ?>"
                               <?php echo $page >= $total_pages ? 'tabindex="-1" aria-disabled="true"' : ''; ?>>
                                Next <i class="bi bi-chevron-right"></i>
                            </a>
                        </li>
                    </ul>
                </nav>
                
                <!-- Quick Jump -->
                <div class="text-center mt-3">
                    <form method="GET" class="d-inline-flex align-items-center gap-2">
                        <?php 
                        // Preserve existing parameters
                        foreach ($_GET as $key => $value) {
                            if ($key !== 'page') {
                                echo '<input type="hidden" name="' . htmlspecialchars($key) . '" value="' . htmlspecialchars($value) . '">';
                            }
                        }
                        ?>
                        <small class="text-muted">Jump to page:</small>
                        <input type="number" name="page" min="1" max="<?php echo $total_pages; ?>" 
                               value="<?php echo $page; ?>" class="form-control form-control-sm" style="width: 80px;">
                        <button type="submit" class="btn btn-sm btn-outline-secondary">Go</button>
                    </form>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- View Application Modal -->
    <div class="modal fade" id="viewModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Application Details</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body" id="applicationDetails">
                    <!-- Content will be loaded via AJAX -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Image Viewer Modal -->
    <div class="modal fade" id="imageViewerModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Image Viewer</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body text-center">
                    <img id="modalImageView" src="" class="img-fluid" style="max-height: 70vh;">
                </div>
                <div class="modal-footer">
                    <a href="#" id="downloadImageBtn" class="btn btn-primary">
                        <i class="bi bi-download"></i> Download
                    </a>
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>
    <script>
        // Update the openDocumentViewer function
        function openDocumentViewer(url, type) {
            if (type === 'image') {
                // Show in modal for images
                $('#modalImageView').attr('src', url);
                $('#downloadImageBtn').attr('href', url);
                $('#imageViewerModal').modal('show');
                return false;
            } else {
                // Open PDFs and other docs in new tab
                window.open(url, '_blank');
                return false;
            }
        }

        function viewApplication(application_id) {
            $.ajax({
                url: '../api/get_application.php',
                type: 'GET',
                data: { application_id: application_id },
                success: function(response) {
                    $('#applicationDetails').html(response);
                }
            });
        }

        function updateStatus(application_id, status) {
            const action = status === 'accepted' ? 'accept' : 'reject';
            
            Swal.fire({
                title: `Confirm ${action}`,
                text: `Are you sure you want to ${action} this application?`,
                icon: 'question',
                showCancelButton: true,
                confirmButtonText: `Yes, ${action}`,
                cancelButtonText: 'Cancel'
            }).then((result) => {
                if (result.isConfirmed) {
                    $.ajax({
                        url: '../api/update_status.php',
                        type: 'POST',
                        data: { application_id, status },
                        dataType: 'json',
                        success: function(response) {
                            Swal.fire({
                                icon: response.success ? 'success' : 'error',
                                title: response.success ? 'Success!' : 'Error',
                                text: response.message
                            }).then(() => {
                                if (response.success) {
                                    location.reload();
                                }
                            });
                        }
                    });
                }
            });
        }

        function promptScore(application_id) {
            Swal.fire({
                title: 'Set Admission Score',
                input: 'number',
                inputAttributes: {
                    min: 0,
                    max: 100,
                    step: 1
                },
                showCancelButton: true,
                confirmButtonText: 'Set Score',
                cancelButtonText: 'Cancel',
                inputValidator: (value) => {
                    if (!value || value < 0 || value > 100) {
                        return 'Please enter a valid score between 0-100';
                    }
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const score = result.value;
                    $.ajax({
                        url: '../api/update_score.php',
                        type: 'POST',
                        data: { application_id, score },
                        dataType: 'json',
                        success: function(response) {
                            Swal.fire({
                                icon: response.success ? 'success' : 'error',
                                title: response.success ? 'Success!' : 'Error',
                                text: response.message
                            }).then(() => {
                                if (response.success) {
                                    location.reload();
                                }
                            });
                        }
                    });
                }
            });
        }

        function uploadLetter(application_id) {
            Swal.fire({
                title: 'Upload Admission Letter',
                html: `<input type="file" id="letterFile" class="form-control">`,
                showCancelButton: true,
                confirmButtonText: 'Upload',
                cancelButtonText: 'Cancel',
                preConfirm: () => {
                    const file = document.getElementById('letterFile').files[0];
                    if (!file) {
                        Swal.showValidationMessage('Please select a file');
                        return false;
                    }
                    return file;
                }
            }).then((result) => {
                if (result.isConfirmed) {
                    const formData = new FormData();
                    formData.append('application_id', application_id);
                    formData.append('admission_letter', result.value);

                    $.ajax({
                        url: '../api/upload_letter.php',
                        type: 'POST',
                        data: formData,
                        processData: false,
                        contentType: false,
                        dataType: 'json',
                        success: function(response) {
                            Swal.fire({
                                icon: response.success ? 'success' : 'error',
                                title: response.success ? 'Success!' : 'Error',
                                text: response.message
                            }).then(() => {
                                if (response.success) {
                                    location.reload();
                                }
                            });
                        }
                    });
                }
            });
        }
    </script>
</body>
</html>
<?php 
$stmt->close();
$count_stmt->close();
$conn->close(); 
?>