<?php
session_start();
require_once '../config/db.php';

if (!isset($_SESSION['admin_id'])) {
    header("Location: admin_login.php");
    exit;
}

$conn = getDB();
$message = '';
$messageType = '';

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (isset($_POST['action'])) {
        switch ($_POST['action']) {
            case 'add_admin':
                $username = trim($_POST['username']);
                $password = trim($_POST['password']);
                
                if (empty($username) || empty($password)) {
                    $message = 'Username and password are required.';
                    $messageType = 'danger';
                } else {
                    // Check if username already exists
                    $check_stmt = $conn->prepare("SELECT id FROM admins WHERE username = ?");
                    $check_stmt->bind_param("s", $username);
                    $check_stmt->execute();
                    $result = $check_stmt->get_result();
                    
                    if ($result->num_rows > 0) {
                        $message = 'Username already exists.';
                        $messageType = 'danger';
                    } else {
                        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                        $stmt = $conn->prepare("INSERT INTO admins (username, password) VALUES (?, ?)");
                        $stmt->bind_param("ss", $username, $hashed_password);
                        
                        if ($stmt->execute()) {
                            $message = 'Admin added successfully.';
                            $messageType = 'success';
                        } else {
                            $message = 'Error adding admin.';
                            $messageType = 'danger';
                        }
                        $stmt->close();
                    }
                    $check_stmt->close();
                }
                break;
                
            case 'update_admin':
                $admin_id = intval($_POST['admin_id']);
                $username = trim($_POST['username']);
                $password = trim($_POST['password']);
                
                if (empty($username)) {
                    $message = 'Username is required.';
                    $messageType = 'danger';
                } else {
                    // Check if username already exists for different admin
                    $check_stmt = $conn->prepare("SELECT id FROM admins WHERE username = ? AND id != ?");
                    $check_stmt->bind_param("si", $username, $admin_id);
                    $check_stmt->execute();
                    $result = $check_stmt->get_result();
                    
                    if ($result->num_rows > 0) {
                        $message = 'Username already exists.';
                        $messageType = 'danger';
                    } else {
                        if (!empty($password)) {
                            // Update both username and password
                            $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                            $stmt = $conn->prepare("UPDATE admins SET username = ?, password = ? WHERE id = ?");
                            $stmt->bind_param("ssi", $username, $hashed_password, $admin_id);
                        } else {
                            // Update only username
                            $stmt = $conn->prepare("UPDATE admins SET username = ? WHERE id = ?");
                            $stmt->bind_param("si", $username, $admin_id);
                        }
                        
                        if ($stmt->execute()) {
                            $message = 'Admin updated successfully.';
                            $messageType = 'success';
                        } else {
                            $message = 'Error updating admin.';
                            $messageType = 'danger';
                        }
                        $stmt->close();
                    }
                    $check_stmt->close();
                }
                break;
                
            case 'delete_admin':
                $admin_id = intval($_POST['admin_id']);
                
                // Prevent deleting the currently logged-in admin
                if ($admin_id == $_SESSION['admin_id']) {
                    $message = 'You cannot delete your own account.';
                    $messageType = 'danger';
                } else {
                    $stmt = $conn->prepare("DELETE FROM admins WHERE id = ?");
                    $stmt->bind_param("i", $admin_id);
                    
                    if ($stmt->execute()) {
                        $message = 'Admin deleted successfully.';
                        $messageType = 'success';
                    } else {
                        $message = 'Error deleting admin.';
                        $messageType = 'danger';
                    }
                    $stmt->close();
                }
                break;
        }
    }
}

// Get all admins
$query = "SELECT id, username FROM admins ORDER BY username";
$result = $conn->query($query);
$admins = [];
while ($row = $result->fetch_assoc()) {
    $admins[] = $row;
}

$conn->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Settings</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.0/font/bootstrap-icons.css">
    <link rel="stylesheet" href="../assets/css/admin.css">
    <style>
        .sidebar {
            width: 250px;
            min-height: 100vh;
            transition: all 0.3s;
        }
        .sidebar-header {
            border-bottom: 1px solid rgba(255, 255, 255, 0.1);
        }
        .nav-link {
            color: rgba(255, 255, 255, 0.8);
            padding: 0.75rem 1rem;
            border-radius: 0.25rem;
            margin: 0.25rem 0.5rem;
            transition: all 0.2s;
        }
        .nav-link:hover, .nav-link.active {
            color: white;
            background-color: rgba(255, 255, 255, 0.1);
        }
        .nav-link i {
            margin-right: 0.5rem;
        }
        .main-content {
            flex: 1;
            background-color: #f8f9fa;
            padding: 20px;
        }
        .admin-card {
            transition: transform 0.2s;
        }
        .admin-card:hover {
            transform: translateY(-2px);
        }
        .current-admin {
            border-left: 4px solid #28a745;
        }
    </style>
</head>
<body>
    <div class="admin-container d-flex">
        <!-- Sidebar -->
        <?php include 'sidebar.php'; ?>

        <!-- Main Content -->
        <div class="main-content">
            <div class="container-fluid py-4">
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <h2>Admin Management</h2>
                    <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addAdminModal">
                        <i class="bi bi-plus-circle"></i> Add New Admin
                    </button>
                </div>

                <?php if ($message): ?>
                <div class="alert alert-<?= $messageType ?> alert-dismissible fade show" role="alert">
                    <?= htmlspecialchars($message) ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
                </div>
                <?php endif; ?>

                <!-- Admins List -->
                <div class="row">
                    <?php foreach ($admins as $admin): ?>
                    <div class="col-md-6 col-lg-4 mb-3">
                        <div class="card admin-card h-100 <?= $admin['id'] == $_SESSION['admin_id'] ? 'current-admin' : '' ?>">
                            <div class="card-body">
                                <div class="d-flex justify-content-between align-items-start mb-3">
                                    <div>
                                        <h5 class="card-title mb-1">
                                            <i class="bi bi-person-circle text-primary"></i>
                                            <?= htmlspecialchars($admin['username']) ?>
                                        </h5>
                                        <?php if ($admin['id'] == $_SESSION['admin_id']): ?>
                                        <small class="text-success">
                                            <i class="bi bi-check-circle"></i> Current User
                                        </small>
                                        <?php endif; ?>
                                    </div>
                                    <div class="dropdown">
                                        <button class="btn btn-sm btn-outline-secondary" type="button" data-bs-toggle="dropdown">
                                            <i class="bi bi-three-dots-vertical"></i>
                                        </button>
                                        <ul class="dropdown-menu">
                                            <li>
                                                <button class="dropdown-item" onclick="editAdmin(<?= $admin['id'] ?>, '<?= htmlspecialchars($admin['username']) ?>')">
                                                    <i class="bi bi-pencil"></i> Edit
                                                </button>
                                            </li>
                                            <?php if ($admin['id'] != $_SESSION['admin_id']): ?>
                                            <li>
                                                <button class="dropdown-item text-danger" onclick="deleteAdmin(<?= $admin['id'] ?>, '<?= htmlspecialchars($admin['username']) ?>')">
                                                    <i class="bi bi-trash"></i> Delete
                                                </button>
                                            </li>
                                            <?php endif; ?>
                                        </ul>
                                    </div>
                                </div>
                                <small class="text-muted">Admin ID: <?= $admin['id'] ?></small>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <!-- Add Admin Modal -->
    <div class="modal fade" id="addAdminModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Add New Admin</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="add_admin">
                        <div class="mb-3">
                            <label for="add_username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="add_username" name="username" required>
                        </div>
                        <div class="mb-3">
                            <label for="add_password" class="form-label">Password</label>
                            <input type="password" class="form-control" id="add_password" name="password" required>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Add Admin</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Admin Modal -->
    <div class="modal fade" id="editAdminModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Edit Admin</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="update_admin">
                        <input type="hidden" name="admin_id" id="edit_admin_id">
                        <div class="mb-3">
                            <label for="edit_username" class="form-label">Username</label>
                            <input type="text" class="form-control" id="edit_username" name="username" required>
                        </div>
                        <div class="mb-3">
                            <label for="edit_password" class="form-label">New Password</label>
                            <input type="password" class="form-control" id="edit_password" name="password">
                            <div class="form-text">Leave blank to keep current password</div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-primary">Update Admin</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Delete Admin Modal -->
    <div class="modal fade" id="deleteAdminModal" tabindex="-1">
        <div class="modal-dialog">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title">Delete Admin</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <form method="POST">
                    <div class="modal-body">
                        <input type="hidden" name="action" value="delete_admin">
                        <input type="hidden" name="admin_id" id="delete_admin_id">
                        <p>Are you sure you want to delete the admin <strong id="delete_admin_name"></strong>?</p>
                        <div class="alert alert-warning">
                            <i class="bi bi-exclamation-triangle"></i>
                            This action cannot be undone.
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                        <button type="submit" class="btn btn-danger">Delete Admin</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>

    <script>
        function editAdmin(adminId, username) {
            document.getElementById('edit_admin_id').value = adminId;
            document.getElementById('edit_username').value = username;
            document.getElementById('edit_password').value = '';
            
            const editModal = new bootstrap.Modal(document.getElementById('editAdminModal'));
            editModal.show();
        }

        function deleteAdmin(adminId, username) {
            document.getElementById('delete_admin_id').value = adminId;
            document.getElementById('delete_admin_name').textContent = username;
            
            const deleteModal = new bootstrap.Modal(document.getElementById('deleteAdminModal'));
            deleteModal.show();
        }

        // Clear form when add modal is closed
        document.getElementById('addAdminModal').addEventListener('hidden.bs.modal', function () {
            document.getElementById('add_username').value = '';
            document.getElementById('add_password').value = '';
        });
    </script>
</body>
</html>